//
//  MockHKHealthStore.swift
//  death_app Watch AppTests
//
//  Created by Task Master on 2025-09-16.
//

import Foundation
import HealthKit
@testable import death_app_Watch_App

class MockHKHealthStore: HKHealthStore {
    
    // Mock data
    var mockSamples: [HKSample] = []
    var mockAuthorizationStatus: HKAuthorizationStatus = .notDetermined
    var shouldSucceedAuthorization = true
    var shouldSucceedBackgroundDelivery = true
    var shouldFailWithNetworkError = false
    var shouldFailWithPermissionError = false
    var authorizationError: Error?
    
    // Tracking
    var stoppedQueries: [HKQuery] = []
    var executedQueries: [HKQuery] = []
    
    // Override authorization methods
    override func requestAuthorization(toShare typesToShare: Set<HKSampleType>?, read typesToRead: Set<HKObjectType>?, completion: @escaping (Bool, Error?) -> Void) {
        
        DispatchQueue.global().asyncAfter(deadline: .now() + 0.1) {
            if self.shouldSucceedAuthorization {
                completion(true, nil)
            } else {
                completion(false, self.authorizationError)
            }
        }
    }
    
    override func authorizationStatus(for type: HKObjectType) -> HKAuthorizationStatus {
        return mockAuthorizationStatus
    }
    
    // Override query execution
    override func execute(_ query: HKQuery) {
        executedQueries.append(query)
        
        DispatchQueue.global().asyncAfter(deadline: .now() + 0.1) {
            if let sampleQuery = query as? HKSampleQuery {
                self.handleSampleQuery(sampleQuery)
            } else if let statisticsQuery = query as? HKStatisticsQuery {
                self.handleStatisticsQuery(statisticsQuery)
            } else if let observerQuery = query as? HKObserverQuery {
                self.handleObserverQuery(observerQuery)
            }
        }
    }
    
    override func stop(_ query: HKQuery) {
        stoppedQueries.append(query)
    }
    
    // Background delivery
    override func enableBackgroundDelivery(for type: HKObjectType, frequency: HKUpdateFrequency, withCompletion completion: @escaping (Bool, Error?) -> Void) {
        
        DispatchQueue.global().asyncAfter(deadline: .now() + 0.1) {
            if self.shouldSucceedBackgroundDelivery {
                completion(true, nil)
            } else {
                completion(false, NSError(domain: "MockError", code: 1, userInfo: nil))
            }
        }
    }
    
    override func disableBackgroundDelivery(for type: HKObjectType, withCompletion completion: @escaping (Bool, Error?) -> Void) {
        
        DispatchQueue.global().asyncAfter(deadline: .now() + 0.1) {
            completion(true, nil)
        }
    }
    
    // Private helper methods
    private func handleSampleQuery(_ query: HKSampleQuery) {
        let relevantSamples = mockSamples.filter { sample in
            return sample.sampleType == query.sampleType &&
                   sample.startDate >= query.predicate?.startDate ?? Date.distantPast &&
                   sample.endDate <= query.predicate?.endDate ?? Date.distantFuture
        }
        
        if shouldFailWithNetworkError {
            let error = NSError(domain: HKErrorDomain, code: 5, userInfo: [NSLocalizedDescriptionKey: "Network unavailable"])
            query.resultsHandler?(query, nil, error)
            return
        }
        
        if shouldFailWithPermissionError {
            let error = NSError(domain: HKErrorDomain, code: 4, userInfo: [NSLocalizedDescriptionKey: "Permission denied"])
            query.resultsHandler?(query, nil, error)
            return
        }
        
        query.resultsHandler?(query, relevantSamples, nil)
    }
    
    private func handleStatisticsQuery(_ query: HKStatisticsQuery) {
        // Create mock statistics
        let mockStatistics = MockHKStatistics()
        query.resultsHandler?(query, mockStatistics, nil)
    }
    
    private func handleObserverQuery(_ query: HKObserverQuery) {
        // Simulate observer query setup
        query.resultsHandler?(query, {
            // Completion handler
        }, nil)
    }
}

// MARK: - Helper Extensions

extension HKSampleQuery {
    var resultsHandler: HKSampleQueryResultsHandler? {
        return nil // This would need to be properly implemented in a real mock
    }
}

extension HKStatisticsQuery {
    var resultsHandler: HKStatisticsQueryResultsHandler? {
        return nil // This would need to be properly implemented in a real mock
    }
}

extension HKObserverQuery {
    var resultsHandler: HKObserverQueryResultsHandler? {
        return nil // This would need to be properly implemented in a real mock
    }
}

extension NSPredicate {
    var startDate: Date? {
        // Extract start date from predicate - simplified implementation
        return nil
    }
    
    var endDate: Date? {
        // Extract end date from predicate - simplified implementation
        return nil
    }
}

// Mock HKStatistics class
class MockHKStatistics: HKStatistics {
    override var averageQuantity: HKQuantity? {
        return HKQuantity(unit: .count(), doubleValue: 75.0)
    }
    
    override var maximumQuantity: HKQuantity? {
        return HKQuantity(unit: .count(), doubleValue: 120.0)
    }
    
    override var minimumQuantity: HKQuantity? {
        return HKQuantity(unit: .count(), doubleValue: 60.0)
    }
    
    override var sumQuantity: HKQuantity? {
        return HKQuantity(unit: .count(), doubleValue: 10000.0)
    }
}